#include "aes.h"
#include <string.h>
#include <stdio.h>
// AES class implementation

#define bindProc(NAME) (NAME##Proc) GetProcAddress(_hdll,#NAME)

static const unsigned __bufsize = 400;

/** \fn AES::AES()
@func Constructor initializes interna string buffer, loads dynamic library
and assigns library function pointers.
*/
AES::AES()
{
   _hdll = LoadLibrary("aes.exe");
   _getVersion = bindProc(getRomVersion);
   _drawBox = bindProc(drawBox);
   _showMenu = bindProc(showMenu);
   _showMessage = bindProc(showMessage);
   _showError = bindProc(showError);
   _getKey = bindProc(getKey);
   _getScrSize = bindProc(getScreenSize);
   _getScrMode = bindProc(getScreenMode);
   _setScrMode = bindProc(setScreenMode);
   _editString = bindProc(editString);
   _getIntDiskSize = bindProc(getInternalDiskSize);
   _getVScreenPos = bindProc(getVScreenPos);
   _setVScreenPos = bindProc(setVScreenPos);
   _getCursorMode = bindProc(getCursorMode);
   _setCursorMode = bindProc(setCursorMode);
   _setCursorPos = bindProc(setCursorPosition);
   _getCursorPos = bindProc(getCursorPosition);
   _setScrPage = bindProc(setScrPage);
   _getScrPage = bindProc(getScrPage);
   _strbuf = new char[__bufsize];
   _strbuf[0] = 0;
   _videoPage = 0;
   _lineStyle = LS_SINGLE;
}

/**
@func Destructor releases internal string buffer and dynamic library
*/
AES::~AES()
{
   FreeLibrary(_hdll);
   delete _strbuf;
}

// get/set helpers

/**
@func Function sets number of video page to be used for boxed output and cursor functions.
@param page page to use
@see getVideoPage, drawBox, showMenu, showMessage, showError, getCursorX, getCursorY, setCursorPos
*/
void AES::setVideoPage(unsigned page)
{
   _videoPage = (page & 0xFF);
}

/**
@return number of video page currently set
@see setVideoPage
*/
unsigned AES::getVideoPage() const
{
   return _videoPage;
}

/**
@param lineStyle line style for use with drawBox function, possible values are
LS_SINGLE (0) and LS_DOUBLE (1)
*/
void AES::setLineStyle(unsigned lineStyle)
{
   _lineStyle = (lineStyle) ? LS_DOUBLE : LS_SINGLE;
}

/**
@return line style for use with drawBox function
@see setLineStyle
*/
unsigned AES::getLineStyle() const
{
   return _lineStyle;
}


/**
@return value of internal string buffer
@note The buffer is owned by AES object, i.e. client code must not delete the returned string by a call to
free function.
*/
const char* AES::getBuffer() const
{
   return _strbuf;
}

/**
@param str string to put into internal buffer
@note Function makes a copy of <i>str</i>, i.e. <i>str</i> will be not changed if internal buffer is modified.
If length of <i>str</i> exceeds size of buffer, value is truncated.
@see __bufsize
*/
void AES::setBuffer(const char* str)
{
   if (str == 0)
   {
	   _strbuf[0] = 0;
	   return;
   }
   unsigned n = strlen(str);
   if (n >= __bufsize) n = __bufsize-1;
   strncpy(_strbuf,str,n);
   _strbuf[n] = 0;
}

/**
@func Function encodes rectangle coordinates into a single long number. It is currently used by
drawBox function.
@param x x-coordinate (column) of upper left corner
@param y y-coordinate (row) of upper left corner
@param width width of rectangle in character columns
@param height height of rectangle in character rows
@return long value containing C-structure as follows:
@code
struct
{
     unsigned char left;    // column of upper left corner
     unsigned char top;     // row of upper left corner
     unsigned char right;   // column of bottom right corner
     unsigned char bottom;  // row of bottom right corner
};
@endcode
*/
long AES::mkRect(int x, int y, unsigned width, unsigned height)
{
	unsigned long result, tmp;
	tmp = y+height;
	result = ((tmp & 0xFF) << 24);
	tmp = x+width;
	result += ((tmp & 0xFF) << 16);
	result += ((y & 0xFF) << 8) + (x & 0xFF);
    return (long) result;
}


/**
@return Version of ROM-BIOS in string representation
@note The returned string is contained in buffer of AES object, i.e. it must not be deleted by client code.
*/
const char* AES::getRomVersion()
{
   _getVersion(__bufsize,_strbuf);
   return _strbuf;
}

/**
@func Function draws a box in text mode on current video page and using current line style.
@param x x-coordinate (column) of upper left corner
@param y y-coordinate (row) of upper left corner
@param width width of rectangle in character columns
@param height height of rectangle in character rows
@note Function preserves position of cursor.
@see setVideoPage, setLineStyle, mkRect
*/
void AES::drawBox(int x, int y, unsigned width, unsigned height)
{
   long style = (_videoPage << 16) + _lineStyle;
   _drawBox(mkRect(x,y,width,height), style);
}

/**
@func Function shows a popup menu on current video page ant waits for selection.
@param x x-coordinate (column) of upper left corner
@param y y-coordinate (row) of upper left corner
@param desc Descriptor of menu contents. The Descriptor simply consists of any number of
lines where first line contains menu title (no title if left blank) ond other lines
are the menu items, e.g. "MyMenu\n1 First item\n2 Second item\n3 Third item".
@param selection This argument specifies index of upper visible item as soon as preselected item when menu is shown.
The lowest byte is index of selected item where high byte of lower word is index of upper line, e.g.
a value of 0x0206 would make item of index 2 to be first visible menu item and item of index 6 preselected one.
@return Result of selection. If user had selected an item returned value has same structure as described for
<i>selection</i> argument. Otherwise (i.e. menu was canceled by ESC key) function returns -1;
@note Function preserves screen area needed by menu as soon as cursor position. The maximum visible height of
popup menu is restricted to 8 lines.
*/
int AES::showMenu(int x, int y, const char* desc, int selection)
{
   char* buf = new char[strlen(desc) + 3];
   sprintf(buf,"%s\n",desc);
   int result = _showMenu(buf,(y << 8) + x, selection, _videoPage);
   delete buf;
   return result;
}

/**
@func Function displays a messagebox of one or more lines on screen and waits until any key pressed.
@param x x-coordinate (column) of upper left corner
@param y y-coordinate (row) of upper left corner
@param msgText Contents of messagebox. The Descriptor simply consists of any number of
lines where first line contains box title (no title if left blank) and other lines
are the lines of the message itselfs.
@return Extended 16-bit-code of key pressed, where low byte contains character code while high byte the scan code.
@note Function preserves screen area needed by messagebox as soon as cursor position.
*/
unsigned AES::showMessage(int x, int y, const char* msgText)
{
   char* buf = new char[strlen(msgText) + 3];
   sprintf(buf,"%s\n",msgText);
   unsigned result = _showMessage(buf,(y << 8) + x, _videoPage);
   delete buf;
   return result;
}

/**
@func Function shows an error message. It is very similary to showMessage function. However box shown has not a title
and a beep is emitted on show.
@param x x-coordinate (column) of upper left corner
@param y y-coordinate (row) of upper left corner
@param msgText Text of error message. The message may (but needs not) consist of multiple lines.
@return Extended 16-bit-code of key pressed to close box, where low byte contains character code while high byte the scan code.
@note Function preserves screen area needed by messagebox as soon as cursor position.
*/
unsigned AES::showError(int x, int y, const char* msgText)
{
   char* buf = new char[strlen(msgText) + 3];
   sprintf(buf,"%s\n",msgText);
   unsigned result = _showError(buf,(y << 8) + x, _videoPage);
   delete buf;
   return result;
}


/**
@func Function allows editing of a string in a separate window at any position. It preserves screen area before
displaying editor and restores area as soon as cursor position at end of editing. String to edit is assumed
to be contained in internal string buffer. Editing itselfs ends if ESC (cancel) or ENTER (apply) key pressed by
user. In case of ENTER modified string is written back to internal buffer, otherwise value of buffer is not changed..
@param x x-coordinate (column) of upper left corner of editor window
@param y y-coordinate (row) of upper left corner of editor window
@param hasBorder specifies if editor should use a border
@param width width of editor window on screen (in charecters including borders)
@param title caption of editor window (shown only if <i>hasBorder</i> is <tt>true</tt>, on empty string no caption is shown
@param prompt prompt string to display left before value to be edited, use empty string if prompt not needed
@return true if editor was closed by ENTER key, otherwise false
@note <ul>
  <li>Maximum length of sring to edit is restricted by value of __bufsize.</li>
  <li>If <i>hasBorder</i> is <tt>true</tt> (i.e. not 0) frame is drawn using style defined by setLineStyle method.</li>
  <li>If a prompt is used, length of prompt text reduces visible space for editing.</li>
  <li>Use getBuffer method to get modified string value.</i>
</ul>
@see setLineStyle, setBuffer, getBuffer, __bufsize
*/
bool AES::editString(int x, int y, unsigned width, bool hasBorder,const char* title, const char* prompt)
{
   long pos = ((y & 0xFF) << 8) + (x & 0xFF);
   unsigned titleSize = strlen(title) + strlen(prompt) +10;
   char* titleBuf = new char[titleSize];
   sprintf(titleBuf,"%s\n%s\n\n",title,prompt);
   long style = (hasBorder) ? _lineStyle : LS_NONE;
   style += ((long)(width & 0xFF)) << 16;
   bool result = (_editString(pos,titleBuf,_strbuf,__bufsize,style) != 0) ? true : false;
   delete titleBuf;
   return result;
}

/**
@func Function waits for pressing a key and returns its extended code.
@return Extended 16-bit-code of key, where low byte contains character code while high byte the scan code.
*/
unsigned AES::getKey() const
{
   return _getKey();
}

/**
@return number of physical screen columns
*/
unsigned AES::getScrW() const
{
   return _getScrSize() & 0xFF;
}

/**
@return number of physical screen rows
*/
unsigned AES::getScrH() const
{
   return (_getScrSize() >> 8) & 0xFF;
}

/**
@return number of logical screen columns
*/
unsigned AES::getLogScrW() const
{
   return (_getScrSize() >> 16) & 0xFF;
}

/**
@return number of logical screen rows
*/
unsigned AES::getLogScrH() const
{
   return (_getScrSize() >> 24) & 0xFF;
}

/**
@return Current mode of Portfolio screen. Result may be one of following values:
<ul>
  <li>SM_STATIC (0) - static 80x25 mode</li>
  <li>SM_POFO (1) - Portfolio native screen mode 40x8</li>
  <li>SM_DYNAMIC (2) - dynamic 80x25 mode</li>
  <li>SM_GRAPHIC (0x80) - graphic mode
</ul>
@see setScrMode
*/
unsigned AES::getScrMode() const
{
   return _getScrMode();
}

/**
@func Function sets screen to specified mode.
@param mode Screen mode to set. Possible values are same as described for getScrMode.
@return old screen mode, i.e. mode set before
@see getScrMode
*/
unsigned AES::setScrMode(unsigned mode)
{
   return _setScrMode(mode);
}

/**
@return size of internal RAM disk in KBytes
*/
unsigned long AES::getInternalDiskSize() const
{
   return _getIntDiskSize();
}

/**
@return logical column position of virtual screens upper left corner
*/
int AES::getVScreenPosX() const
{
   return (_getVScreenPos() & 0xFF);
}


/**
@return logical row position of virtual screens upper left corner
*/
int AES::getVScreenPosY() const
{
   return ((_getVScreenPos() >> 8) & 0xFF);
}

/**
@param x new logical column position of virtual screens upper left corner
@param y new logical row position of virtual screens upper left corner
@note This function should be used only if current screen mode is SM_STATIC od SM_DYNAMIC
@see getScreenMode
*/
void AES::setVScreenPos(int x,int y)
{
   _setVScreenPos(x,y);
}

/**
@return Current mode of cursor. Result may be one of following values:
<ul>
  <li>CM_OFF (0) - cursor not visible</li>
  <li>CM_UNDERLINE (1) - underline cursor</li>
  <li>CM_BLOCK (2) - block cursor</li>
</ul>
@see setCursorMode
*/
unsigned AES::getCursorMode() const
{
   return _getCursorMode();
}

/**
@func Function sets cursor to specified mode.
@param mode Cursor mode to set, following values are valid:
<ul>
  <li>CM_OFF (0) - cursor not visible</li>
  <li>CM_UNDERLINE (1) - underline cursor</li>
  <li>CM_BLOCK (2) - block cursor</li>
  <li>CM_REFLECT (3) - set cursor mode to reflect keyboard NumLock state
</ul>
@return old cursor mode, i.e. mode set before
@see getCursorMode
*/
unsigned AES::setCursorMode(unsigned mode)
{
   return _setCursorMode(mode);
}

/**
@func Function reads x-coordinate of cursor for video page specified by #_videoPage attribute.
@return x- (column) position of cursor
@see getCursorY, setCursorPos, setVideoPage
*/
int AES::getCursorX() const
{
   return (_getCursorPos(_videoPage) & 0xFF);
}

/**
@func Function reads x-coordinate of cursor for video page specified by #_videoPage attribute.
@return y- (row) position of cursor
@see getCursorX, setCursorPos, setVideoPage
*/
int AES::getCursorY() const
{
   return ((_getCursorPos(_videoPage) >> 8) & 0xFF);
}

/**
@func Function sets cursor position for video page specified by #_videoPage attribute.
@param x column position for cursor
@param y row position for cursor
@note Application should not set cursor outside of logical screen.
@see getCursorX, getCursorY, setVideoPage
*/
void AES::setCursorPos(int x, int y)
{
   _setCursorPos(x,y,_videoPage);
}

   /// make video page active
/**
@func Function makes specified page to active (i.e. visible) page on screen
@param page number of page to activate
@note Valid values for <i>page</i> depend on video adapter used. For Atari Portfolio pages 0 and 1 should work.
@see setVideoPage, getActiveScrPage
*/
void AES::activateScrPage(unsigned page)
{
   _setScrPage(page);
}

/**
@return number of video page currently active (i.e. visible on screen)
@note Result is read directly from BIOS at address 0:0462H.
@see getVideoPage, activateScrPage
*/
unsigned AES::getActiveScrPage() const
{
   return _getScrPage();
}
