
#include "aes.h"

#include <dos.h>
#include <stdio.h>
#include <string.h>

struct Point
{
   Point(int ii, int jj) : i(ii),j(jj),next(0) {}
   ~Point() { if (next) delete next; }
   Point* next;
   int i;
   int j;
};


class PointList
{
public:
   PointList() : _first(0), _last(0) {}
   ~PointList()  { if (_first) delete _first; }
   Point* first() const { return _first; }
   void append(int i,int j);
   void clear();
private:
   Point* _first;
   Point* _last;
};


void PointList::append(int i, int j)
{
   Point* pt = new Point(i,j);
   if (_first) _last->next = pt;
   else _first = pt;
   _last = pt;
}

void PointList::clear()
{
   if (_first)
   {
      delete _first;
      _first = 0;
      _last = 0;
   }
}

class OuterField
{
public:
   OuterField(AES& aes);
   ~OuterField();
   void clear();
   void read();
   void draw();
   bool validate();
   void cleanCrit();
   int evaluateNumber(int i,int j);
   int getAlternative(int i, int j, int v);
   bool completeIt(Point* pt);
   bool solve();
private:
   int** _rows; // Vektor der Zeilen
   int* _crit; // Feld "kritischer Werte"
   int* _ev; // Evaluierungsfeld
   PointList _undecided; // undecided list (PointList)
   AES& _aes;
};

OuterField::OuterField(AES& aes)
:_aes(aes)
{
   _rows = new int*[9];
   _crit = new int[9];
   _ev = new int[9];
   for (int i=0;i<9;++i)
   {
     int* r= new int[9];
     _rows[i] = r;
     for (int j=0;j<9;++j)
     {
        r[j]=0;
     }
   }
}


OuterField::~OuterField()
{
   for (int i=0;i<9;i++)
   {
     delete _rows[i];
   }
   delete _rows;
   delete _crit;
   delete _ev;
}

void OuterField::clear()
{
   for (int i=0;i<9;++i)
   {
      int* r = _rows[i];
      for (int j=0;j<9;++j) r[j] = 0;
   }
}

static REGS regs;
static char getCurChar()
{
	regs.x.ax = 0x800;
	regs.h.bh = 0;
	int86(0x10,&regs,&regs);
	return regs.h.al;
}

void OuterField::read()
{
   int sx = _aes.getCursorX();
   int sy = _aes.getCursorY();
   int y=1;
   for (int i=0;i<9;++i,++y)
   {
      int* r = _rows[i];
      if (y % 4 == 0) ++y;
      int x = 1;
      for (int j=0;j<9;++j,++x)
      {
         if (x % 4 == 0) ++x;
         _aes.setCursorPos(x,y);
         char m = getCurChar();
         if ((m<'1') || (m>'9'))
         {
             r[j] = 0;
         }
         else
         {
             r[j] = m - '0';
         }
      }
   }
   _aes.setCursorPos(sx,sy);
}

void OuterField::draw()
{
   int sx = _aes.getCursorX();
   int sy = _aes.getCursorY();
   _aes.drawBox(0,0,12,12);
   int i;
   for (i=1;i<12;++i)
   {
      _aes.setCursorPos(1,i);
      printf("   |   |");
   }
   _aes.setCursorPos(1,4);
   printf("-----------");
   _aes.setCursorPos(1,8);
   printf("-----------");

   int y=1;
   for (i=0;i<9;++i,++y)
   {
      int* r = _rows[i];
      if (y % 4 == 0) ++y;
      int x = 1;
      for (int j=0;j<9;++j,++x)
      {
         if (x % 4 == 0) ++x;
         _aes.setCursorPos(x,y);
         char m = r[j];
         putc((m ? m + '0' : '*'),stdout);
      }
   }
   _aes.setCursorPos(15,1),
   printf("1 .. 9      : Number");
   _aes.setCursorPos(15,2),
   printf("*           : Empty");
   _aes.setCursorPos(15,3),
   printf("Up,Dn,Rt,Lt : Move");
   _aes.setCursorPos(15,4),
   printf("R, r        : Reset");
   _aes.setCursorPos(15,5),
   printf("Enter       : Solve");
   _aes.setCursorPos(15,6),
   printf("ESC         : Quit");

   _aes.setCursorPos(sx,sy);
}


void OuterField::cleanCrit()
{
   for (int i=0;i<9;++i) _crit[i] = 0;
}

bool OuterField::validate()
{
   bool ok = true;
   int i,j,k,v;
   // Zeilen
   for (i=0;ok && (i<9);++i)
   {
      cleanCrit();
      int* r = _rows[i];
      for (j=0;j<9;++j)
      {
         v = r[j];
         if (v != 0)
         {
            if (_crit[v-1] != 0)
            {
               ok = false;
               break;
            }
            else
               _crit[v-1] = 1;
         }
      }
   }
   // Spalten
   for (j=0;ok && (j<9);++j)
   {
      cleanCrit();
      for (i=0;i<9;++i)
      {
         v = _rows[i][j];
         if (v != 0)
         {
            if (_crit[v-1] != 0)
            {
               ok = false;
               break;
            }
            else
               _crit[v-1] = 1;
         }
      }
   }
   // Dreier
   for (k=0;ok && (k<9);k++)
   {
      int l = 3 * (k / 3);    // Startzeile Dreierfeld
      int m = 3 * (k % 3);  // Startspalte Dreierfeld
      cleanCrit();
       for (i=0;ok && (i<3);++i)
       {
          int* r = _rows[l + i];
          for (j=0;j<3;++j)
          {
             v = r[m+j];
             if (v != 0)
             {
                if (_crit[v-1] != 0)
                {
                   ok = false;
                   break;
                }
                else
                   _crit[v-1] = 1;
             }
          }
       }
   }
   return ok;
}

/**
Zu einem aktuellen Spielstand werden zur Position (i,j) die mglichen Belegungen
in ev abgelegt. Die mglichen Alternativen kommen in aufsteigender Reihenfolge.
Es wird die Anzahl der mglichen Belegungen (also der gltigen Eintrge in _ev) zurckgegeben..
*/
int OuterField::evaluateNumber(int i,int j)
{
   int cnt = 0;
   cleanCrit();
   int* r = _rows[i];
   int l,m,v;
   for (l=0;l<9;++l)
   {
      v=r[l];
      if (v>0)
         _crit[v-1] = 1;
      v=_rows[l][j];
      if (v > 0)
         _crit[v-1] = 1;
   }
   l = i - (i % 3); // Startzeile Dreierfeld
   m = j - (j % 3); // Startspalte Dreierfeld
   for (int ii=0;ii<3;++ii)
   {
      r = _rows[l + ii];
      for (int jj=0;jj<3;++jj)
      {
         v = r[m+jj];
         if (v > 0) _crit[v-1] = 1;
      }
   }
   for (l = 0;l<9;++l)
   {
      if (_crit[l] == 0)
      {
         _ev[cnt] = l+1;
         ++cnt;
      }
   }
   return cnt;
}

/**
Zu einer Position (i,j) wird eine alternative Belegung ermittelt,
die grer als v ist.
Gibt es eine solche Belegung, wird sie zurckgegeben, sonst 0
*/
OuterField::getAlternative(int i,int j,int v)
{
   int n = evaluateNumber(i,j);
   int x;
   for(int k=0;k<n;++k)
   {
      x = _ev[k];
      if (x > v) return x;
   }
   return 0;
}

bool OuterField::completeIt(Point* pt)
{
   int lastValue = 0;
   if (pt)
   {
      int lastI = pt->i;
      int lastJ = pt->j;
      int* r = _rows[lastI];
      while (1)
      {
         lastValue = r[lastJ];
         if (lastValue != 0)
            r[lastJ] = 0;
         lastValue = getAlternative(lastI,lastJ,lastValue);
         if (lastValue > 0)
         {
            r[lastJ] = lastValue;
            if(completeIt(pt->next)) return true; // Eine Komplettierung war mglich.
         } // Komplettierung war nicht mglich nchste Alternative versuchen.
         else return false; // Keine Belegung mglich, gehe eine Level hher.
      }
   }
   return true; // Ich bin erfolgreich hinter das Ende von undecided gekommen.
}

bool OuterField::solve()
{
   bool succ = true;
   int i,j,v;
   while (succ)
   {
      succ = false;
      for (i=0;i<9;++i)
      {
         int* r = _rows[i];
         for (j=0;j<9;++j)
         {
            v = r[j];
            if (v == 0)
            {
               if (evaluateNumber(i,j) == 1)
               {
                  r[j] = _ev[0];
                  succ = true;
               }
            }
         }
      }
   } // end while
   for (i=0;i<9;++i)
   {
      int* r = _rows[i];
      for (j=0;j<9;++j)
      {
         if (r[j] == 0) _undecided.append(i,j);
      }
   }
   succ = completeIt(_undecided.first());
   _undecided.clear();
   return succ;
}

class SudokuApp
{
public:
   SudokuApp();
   ~SudokuApp();
   void run();
private:
   void clearScreen();
   void mainLoop();
   bool _isPofo;
   AES _aes;
   unsigned  _savedScrMode;
   unsigned _savedCurMode;
   OuterField* _field;

};

SudokuApp::SudokuApp()
{
   _aes.getRomVersion();
   _isPofo = (strcmp(_aes.getBuffer(),"1.020") ? true : false);
   _savedScrMode = _aes.getScrMode();
   _savedCurMode = _aes.getCursorMode();
   _aes.setLineStyle(LS_DOUBLE);
   _field = new OuterField(_aes);
}

SudokuApp::~SudokuApp()
{
   clearScreen();
   _aes.setScrMode(_savedScrMode);
   _aes.setCursorMode(_savedCurMode);
   delete _field;
}


void SudokuApp::clearScreen()
{
   unsigned mode = _aes.getScrMode();
   if (_isPofo)
     regs.h.al = 7;
   else
     regs.h.al = 2;
   regs.h.ah = 0;
   int86(0x10,&regs,&regs);
   _aes.setScrMode(mode);
}

void SudokuApp::mainLoop()
{
   int x=1;
   int y=1;
   while (1)
   {
      _aes.setCursorPos(x,y);
      int k = _aes.getKey();
      int c = k & 0xFF;
      int s = k >> 8;
      if (c == 27)
        break;
      if (((c >= '1') && (c <= '9')) || (c == '*'))
      {
         putc(c,stdout);
      }
      else if (s == 72)  // up
      {
         --y;
         if (y % 4 == 0) --y;
         if (y < 1) y = 11;
      }
      else if (s == 75)  // left
      {
         --x;
         if (x % 4 == 0) --x;
         if (x < 1) x = 11;
      }
      else if (s == 77) // right
      {
         ++x;
         if (x % 4 == 0) ++x;
         if (x > 11) x = 1;
      }
      else if (s == 80) // down
      {
         ++y;
         if (y % 4 == 0) ++y;
         if (y > 11) y = 1;
      }
      else if ((c == 'R') || (c == 'r'))
      {
         _field->clear();
         _field->draw();
      }
      else if (c == 13)
      {
         _field->read();
         _aes.setCursorPos(0,0);
         _aes.setCursorPos(1,1);
        if (!_field->validate())
         {
            _aes.showError(3,2,"Invalid field.");
            continue;
         }
         if (_field->solve())
         {
            _field->draw();
            _aes.showMessage(3,2,"\nSudoku solved.");
         }
         else
            _aes.showError(3,2,"No solution found.");
      }
   }
}

void SudokuApp::run()
{
   _aes.setScrMode(SM_DYNAMIC);
   _aes.setVideoPage(0);
   clearScreen();
   _field->draw();
   mainLoop();
}

int main(int,char**)
{
	SudokuApp app;
	app.run();
    return 0;
}
